import os
import sys
import inspect
import traceback

# get the relative path of iw home from this script
script_name = inspect.getfile(inspect.currentframe())
# set infoworks python path in order to allow imports
current_dir = os.path.dirname(os.path.abspath(script_name))
iw_installer_dir = os.path.abspath(os.path.join(current_dir, '..'))
iw_installer_libs_dir = os.path.abspath(
    os.path.join(iw_installer_dir, 'resources', 'python-libs'))
sys.path.insert(0, iw_installer_dir)
sys.path.insert(1, iw_installer_libs_dir)

from core import logger
from core.iw_utils import IWUtils
from core.iw_constants import ServiceNames

# setup logger

logger.setup_logger(IWUtils.get_upgrade_log_file_path(),
                    IWUtils.get_install_log_config_path_upgrade())

import logging
import argparse
import taskflow.engines

from tqdm import tqdm

from taskflow import retry, states
from taskflow.types import notifier
from taskflow.persistence import models
from taskflow.patterns import linear_flow as lf, graph_flow as gf

from common.utils.utils import Utils
from common.utils.tqdm_logger import TqdmLogger

from common.tasks.check_prerequisites_upgrade import CheckPrerequisitesUpgrade
from common.tasks.download_and_extract_iw_binary import DownloadAndExtractIWBinary
from common.tasks.backup_iw_data import BackupIWData
from common.tasks.check_services_status import CheckServicesStatus
from common.tasks.cleanup_and_recreate_hdfs_cache import CleanupAndRecreateHDFSCache
from common.tasks.copy_binaries import CopyBinaries
from common.tasks.migrate_metastore import MigrateMetastore
from common.tasks.index_metastore import IndexMetastore
from common.tasks.refresh_kerberos_ticket import RefreshKerberosTicket
from common.tasks.start_services import StartServices
from common.tasks.stop_services import StopServices
from common.tasks.upgrade_orchestrator_engine import UpgradeOrchestratorEngine
from common.tasks.upgrade_metastore import UpgradeMetastore
from common.tasks.post_upgrade_checks import PostUpgradeChecks
from common.tasks.migrate_configs import MigrateConfigs
from common.tasks.copy_version_files import CopyVersionFiles
from common.tasks.migrate_postgres_metastore import MigratePostgresMetastore
from common.tasks.merge_configs import MergeConfigs
from common.tasks.migrate_env_script import MigrateEnvScript
from common.tasks.migrate_platform_configs import MigratePlatformConf
from common.tasks.upgrade_migrations import UpgradeMigrations
from common.tasks.add_dt_classpath_prereq import AddDTClasspathPrereq
from common.utils.utils import CheckPrerequisites


logger = logging.getLogger(__name__)

bar = None
task_completed = 0
current_iw_version = None
to_iw_version = None
certificate_check = None


def initialize_progressbar(num_tasks):
    global bar
    tqdm_out = TqdmLogger(logger, level=logging.INFO)
    bar = tqdm(total=num_tasks,
               ncols=100,
               desc='Overall Progress',
               file=tqdm_out)


def change_state(flow_detail):
    # TODO: add further filters to modify the state only for the past run
    for task_detail in flow_detail:
        if task_detail.state == states.REVERTED:
            task_detail.reset(states.PENDING)
            flow_detail.state = states.PENDING


def task_transition(state, details):
    global task_completed
    if 'task_name' in details:
        if state in [states.RUNNING]:
            logger.info('Starting task: {}'.format((details['task_name'])))
        if state in [states.SUCCESS]:
            bar.update(1)
            logger.info('Task: {} finished successfully'.format(
                (details['task_name'])))
        if state in [states.REVERTED]:
            logger.info('Task: {} reverted'.format((details['task_name'])))


def get_service_list():
    return [
        'ui',
        'governor',
        'hangman',
        'restapi',
        'queryservice',
        'df',
        'cube-engine',
        'monitor',
        'platform',
        'configuration-service',
        'postgres',
        'rabbitmq',
        'nginx'
    ]

def create_flow(**kwargs):

    services_string = None
    if not args.include_services and not args.exclude_services:
        services_string = "all"
    elif args.include_services:
        service_list = []
        for service in args.include_services:
            service_list.append(ServiceNames.get_name(key=service))
        services_string = ' '.join(service_list)
    elif args.exclude_services:
        service_list = get_service_list()
        for service in args.exclude_services:
            service_list.remove(ServiceNames.get_name(key=service))
        services_string = ' '.join(service_list)
    if not services_string:
        services_string = "all"

    check_prerequisites_flow = lf.Flow(name='Check Prerequisites').add(CheckPrerequisitesUpgrade('Check Prerequisites'))
    download_iw_binary_flow = lf.Flow(name='Download and Extract IW Binary').add(DownloadAndExtractIWBinary('Download and Extract IW Binary', to_iw_version, certificate_check))
    backup_iw_data_flow = lf.Flow(name='Backup IW data').add(BackupIWData('Backup IW data'))
    copy_binaries_flow = lf.Flow(name='Copy Binaries').add(CopyBinaries('Copy Binaries'))
    check_services_status_flow = lf.Flow(name='Check Services Status').add(CheckServicesStatus('Check Services Status'))
    start_services_flow = lf.Flow(name='Start IW Services').add(StartServices('Start IW Services', services_string))
    start_orchestrator_flow = lf.Flow(name='Start Orchestrator Services').add(StartServices('Start Orchestrator Services', 'orchestrator'))
    start_mongo_flow = lf.Flow(name='Start MongoDB').add(StartServices('Start MongoDB', 'mongo'))
    stop_services_flow = lf.Flow(name='Stop IW Services').add(StopServices('Stop IW Services', 'all orchestrator mongo'))
    migrate_configs_flow = lf.Flow(name='Migrate IW Config Files').add(MigrateConfigs('Migrate IW Config Files'))
    migrate_metastore_flow = lf.Flow(name='Migrate Metastore').add(MigrateMetastore('Migrate Metastore'))
    upgrade_metastore_flow = lf.Flow(name='Upgrade Metastore').add(UpgradeMetastore('Upgrade Metastore', '4.0', certificate_check))
    upgrade_orchestrator_engine_flow = lf.Flow(name='Upgrade Orchestrator Engine').add(UpgradeOrchestratorEngine('Upgrade Orchestrator Engine', '1.10.1'))
    post_upgrade_checks_flow = lf.Flow(name='Post Upgrade Checks').add(PostUpgradeChecks('Post Upgrade Checks', to_iw_version))
    index_metastore_flow = lf.Flow(name='Index Metastore').add(IndexMetastore('Index Metastore'))
    copy_version_files_flow = lf.Flow(name='Copy Version Files').add(CopyVersionFiles('Copy Version Files'))
    cleanup_and_recreate_hdfs_cache_flow = lf.Flow(name='Cleanup and Recreate HDFS Cache').add(CleanupAndRecreateHDFSCache('Cleanup and Recreate HDFS Cache'))
    refresh_kerberos_ticket_flow = lf.Flow(name='Refresh Kerberos Ticket').add(RefreshKerberosTicket('Refresh Kerberos Ticket'))
    check_services_status_post_flow = lf.Flow(name='Check Post Upgrade Services Status').add(CheckServicesStatus('Check Post Upgrade Services Status'))
    start_config_service_flow = lf.Flow(name ='Start configuration service to merge config files').add(StartServices('Start Configuration Service', 'configuration-service'))
    merge_configs_flow = lf.Flow(name = 'Merging Configuration files').add(MergeConfigs('Merge Configuration',current_iw_version, to_iw_version))
    migrate_postgres_metastore_flow = lf.Flow(name='Migrate Postgres Metastore').add(MigratePostgresMetastore('Migrate Postgres Metastore'))
    migrate_env_script_flow = lf.Flow(name='Migrate Env Script').add(MigrateEnvScript('Migrate Env Script'))
    migrate_platform_configs_flow = lf.Flow(name='Migrate Platform Configs').add(MigratePlatformConf('Migrate Platform Configs', current_iw_version))
    add_dt_classpath_prereq_flow = lf.Flow(name='DT Classpath Updates').add(AddDTClasspathPrereq('DT Classpath Updates'))
    upgrade_migrations_flow = lf.Flow(name='Migrations for Upgrade').add(UpgradeMigrations('Migrations for Upgrade'))

    flow = gf.Flow(name='Infoworks Upgrade Flow')\
        .add(check_prerequisites_flow) \
        .add(stop_services_flow) \
        .add(download_iw_binary_flow) \
        .add(backup_iw_data_flow) \
        .add(start_mongo_flow) \
        .add(upgrade_metastore_flow) \
        .add(check_services_status_flow)\
        .add(copy_binaries_flow) \
        .add(migrate_configs_flow) \
        .add(migrate_metastore_flow) \
        .add(migrate_postgres_metastore_flow) \
        .add(index_metastore_flow) \
        .add(copy_version_files_flow) \
        .add(refresh_kerberos_ticket_flow) \
        .add(cleanup_and_recreate_hdfs_cache_flow) \
        .add(start_services_flow)\
        .add(upgrade_orchestrator_engine_flow) \
        .add(start_orchestrator_flow) \
        .add(post_upgrade_checks_flow)\
        .add(check_services_status_post_flow) \
        .add(start_config_service_flow) \
        .add(merge_configs_flow) \
        .add(migrate_env_script_flow) \
        .add(migrate_platform_configs_flow) \
        .add(add_dt_classpath_prereq_flow) \
        .add(upgrade_migrations_flow)

    flow.link(check_prerequisites_flow, stop_services_flow)
    flow.link(stop_services_flow, check_services_status_flow)
    flow.link(check_services_status_flow, download_iw_binary_flow)
    flow.link(check_services_status_flow, backup_iw_data_flow)
    flow.link(download_iw_binary_flow, copy_binaries_flow)
    flow.link(download_iw_binary_flow, copy_version_files_flow)
    flow.link(backup_iw_data_flow, copy_binaries_flow)
    flow.link(backup_iw_data_flow, copy_version_files_flow)
    flow.link(copy_binaries_flow, migrate_env_script_flow)
    flow.link(migrate_env_script_flow, start_mongo_flow)
    flow.link(start_mongo_flow, start_config_service_flow)
    flow.link(start_mongo_flow, migrate_platform_configs_flow)
    flow.link(start_config_service_flow, merge_configs_flow)
    flow.link(merge_configs_flow, upgrade_metastore_flow)
    flow.link(copy_version_files_flow, upgrade_metastore_flow)
    flow.link(upgrade_metastore_flow, migrate_configs_flow)
    flow.link(upgrade_metastore_flow, refresh_kerberos_ticket_flow)
    flow.link(migrate_configs_flow, migrate_metastore_flow)
    flow.link(migrate_metastore_flow, index_metastore_flow)
    flow.link(migrate_metastore_flow, start_services_flow)
    flow.link(refresh_kerberos_ticket_flow, cleanup_and_recreate_hdfs_cache_flow)
    flow.link(cleanup_and_recreate_hdfs_cache_flow, start_services_flow)
    flow.link(migrate_metastore_flow, upgrade_migrations_flow)
    flow.link(index_metastore_flow, start_services_flow)
    flow.link(upgrade_migrations_flow, start_services_flow)
    flow.link(start_services_flow, migrate_postgres_metastore_flow)
    flow.link(migrate_postgres_metastore_flow, upgrade_orchestrator_engine_flow)
    flow.link(upgrade_orchestrator_engine_flow, start_orchestrator_flow)
    flow.link(start_orchestrator_flow, check_services_status_post_flow)
    flow.link(check_services_status_post_flow, add_dt_classpath_prereq_flow)
    flow.link(add_dt_classpath_prereq_flow, post_upgrade_checks_flow)

    return flow


def run_upgrade(restart=False, uuid=None):
    Utils.get_connection().upgrade()
    book = models.LogBook('Infoworks Upgrade Flow', uuid=to_iw_version)
    store = {
        'to_iw_version': to_iw_version,
        'certificate_check': certificate_check
    }

    if not restart:
        flow_detail = models.FlowDetail('Infoworks Upgrade', uuid=to_iw_version)
        book.add(flow_detail)
        Utils.get_connection().save_logbook(book)

    else:
        flow_detail = Utils.find_flow_detail(Utils.get_backend(), to_iw_version,
                                             to_iw_version)
        change_state(flow_detail)
        Utils.get_connection().update_flow_details(flow_detail)

    flow = create_flow()

    engine = taskflow.engines.load(flow,
                                   flow_detail=flow_detail,
                                   store=store,
                                   engine='parallel',
                                   backend=Utils.get_backend(),
                                   book=book)

    initialize_progressbar(flow.__len__())
    engine.atom_notifier.register(notifier.Notifier.ANY, task_transition)
    taskflow.engines.save_factory_details(flow_detail=flow_detail,
                                          flow_factory=create_flow,
                                          factory_args=[],
                                          factory_kwargs=store,
                                          backend=Utils.get_backend())

    run_success = True
    try:
        engine.run()
    except Exception as e:
        logger.exception(e)
        with open(IWUtils.get_upgrade_log_file_path(), 'a') as fp:
            traceback.print_exc(file=fp)
        logger.info(
            '************************************************ WARNING ************************************************'
        )
        logger.info('The system might be in an inconsistent state. '
                    'Please contact Infoworks support for further assistance')
        logger.info(
            '*********************************************************************************************************'
        )
        run_success = False
    finally:
        bar.close()
        if not run_success:
            sys.exit(1)


if __name__ == '__main__':
    # arg parse
    parser = argparse.ArgumentParser('*** Infoworks Upgrade ***')
    parser.add_argument('-r',
                        '--restart',
                        dest='restart',
                        action='store_true',
                        help='Restart an existing upgrade process')

    parser.add_argument('--certificate-check',
                        dest='certificate_check',
                        help='Security Certificate Check (true/false)',
                        default="true")

    parser.add_argument('--include-services', dest='include_services',
                        help='List of all services to start during install',
                        default=None, nargs='+')

    parser.add_argument('--exclude-services', dest='exclude_services',
                        help='List of all services to not start during install. Any service not mentioned here will be started. Use either this option or include-service. Do not use both',
                        default=None, nargs='+')

    required_argument = parser.add_argument_group('required arguments')
    required_argument.add_argument(
        '-v',
        '--version',
        dest='version',
        nargs=1,
        help='Infoworks version to upgrade to. Example: 2.7.0',
        required=True)

    args = parser.parse_args()
    current_iw_version = IWUtils.get_iw_version()
    to_iw_version = args.version[0]

    if args.include_services and args.exclude_services:
        raise RuntimeError("Use either include-services parameter or exclude-services parameter. Do not use both")

    certificate_check = args.certificate_check.lower()
    if certificate_check == "true":
        certificate_check = True
    else:
        certificate_check = False

    logger.info(
        'Starting Infoworks upgrade to version {}'.format(to_iw_version))
    logger.info('Current Infoworks version: {}'.format(current_iw_version))

    CheckPrerequisites.is_user_authorized()

    run_upgrade(restart=args.restart)
